
//+------------------------------------------------------------------+
//|                                           Gold_Scalping_MillX.mq5 |
//|                             Copyright 2025, Gold Scalping Bot   |
//|                                     Based on Millionaire X System |
//+------------------------------------------------------------------+
#property copyright "Copyright 2025, Gold Scalping Bot"
#property link      "Based on Millionaire X System"
#property version   "1.00"
#property description "Gold Scalping Bot with RSI Filter and Martingale System"

//--- Include necessary libraries
#include <Trade\Trade.mqh>

//+------------------------------------------------------------------+
//|                           ENUMERATIONS                           |
//+------------------------------------------------------------------+

enum ENUM_RISK_LEVEL
{
   RISK_LOW = 0,        // Low Risk
   RISK_MEDIUM = 1,     // Medium Risk  
   RISK_HIGH = 2,       // High Risk
   RISK_VERY_HIGH = 3   // Very High Risk
};

//--- Global variables
CTrade trade;

//+------------------------------------------------------------------+
//|                           INPUT PARAMETERS                        |
//+------------------------------------------------------------------+

//--- EA Settings
input group "=== EA SETTINGS ==="
input string    EA_Comment = "Gold_Scalp_MillX";         // EA Comment
input int       Magic_Number = 280;                      // Magic Number
input ENUM_TIMEFRAMES Trading_Timeframe = PERIOD_M1;     // Trading Timeframe

//--- RSI Settings  
input group "=== RSI SETTINGS ==="
input bool      Use_RSI_Filter = true;                   // Use RSI as Filter
input int       RSI_Period = 14;                         // RSI Period
input ENUM_APPLIED_PRICE RSI_Applied_Price = PRICE_CLOSE;// RSI Applied Price
input double    RSI_Overbought = 75.0;                  // RSI Overbought Level
input double    RSI_Oversold = 35.0;                    // RSI Oversold Level

//--- Risk Management Settings
input group "=== RISK MANAGEMENT ==="
input ENUM_RISK_LEVEL Risk_Level = RISK_LOW;             // Risk Level
input double    Min_Distance_Points = 2000.0;           // Min Distance Between Levels (Points)
input double    Take_Profit_Points = 1000.0;            // Take Profit (Points)
input double    Stop_Loss_Points = 100000.0;            // Stop Loss (Points) - 0 = No SL
input int       Max_Open_Positions = 20;                // Maximum Open Positions
input double    Max_Slippage_Points = 20;               // Maximum Slippage (Points)

//--- Trading Hours
input group "=== TRADING HOURS ==="
input bool      Use_Trading_Hours = true;               // Enable Trading Hours
input int       Start_Hour = 0;                         // Start Hour (0-23)
input int       End_Hour = 23;                          // End Hour (0-23)

//--- Safety Settings
input group "=== SAFETY SETTINGS ==="
input bool      Enable_Max_DD_Pause = false;            // Enable Max Drawdown Pause
input double    Max_DD_Stop_Percent = 10.0;            // Max Drawdown Stop (%)
input bool      Use_News_Filter = true;                 // Avoid High Impact News

//+------------------------------------------------------------------+
//|                        GLOBAL VARIABLES                          |
//+------------------------------------------------------------------+

// RSI Handle
int rsi_handle;

// Lot size arrays for different risk levels
double lot_sizes_low[] = {0.01,0.02,0.03,0.05,0.08,0.11,0.17,0.23,0.30,0.38,0.47,0.57,0.68,0.80,0.93,1.07,1.22,1.38,1.55,1.73,1.92,2.12};
double lot_sizes_medium[] = {0.03,0.06,0.09,0.15,0.24,0.35,0.51,0.69,0.90,1.14,1.41,1.71,2.04,2.40,2.79,3.21,3.66,4.14,4.65,5.19,5.76,6.36};
double lot_sizes_high[] = {0.06,0.10,0.16,0.26,0.42,0.68,1.10};
double lot_sizes_very_high[] = {0.09,0.18,0.27,0.45,0.72,1.05,1.53,2.07,2.70,3.42,4.23,5.13,6.12,7.20,8.37,9.63,10.98,12.42,13.95,15.57,17.28,19.08};

// Trading variables
double last_buy_price = 0;
double last_sell_price = 0;
int current_buy_level = 0;
int current_sell_level = 0;
bool trading_allowed = true;
datetime last_trade_time = 0;

//+------------------------------------------------------------------+
//|                    EXPERT INITIALIZATION                         |
//+------------------------------------------------------------------+
int OnInit()
{
   // Set magic number for trade operations
   trade.SetExpertMagicNumber(Magic_Number);
   trade.SetDeviationInPoints((ulong)Max_Slippage_Points);

   // Initialize RSI indicator
   rsi_handle = iRSI(_Symbol, Trading_Timeframe, RSI_Period, RSI_Applied_Price);

   if(rsi_handle == INVALID_HANDLE)
   {
      Print("Failed to create RSI indicator handle");
      return(INIT_FAILED);
   }

   // Validate symbol
   if(_Symbol != "XAUUSD" && _Symbol != "GOLD")
   {
      Print("Warning: This EA is designed for Gold (XAUUSD) trading");
   }

   // Initialize variables
   ResetTradingVariables();

   Print("Gold Scalping MillX EA initialized successfully");
   Print("Symbol: ", _Symbol);
   Print("Risk Level: ", GetRiskLevelString());
   Print("RSI Filter: ", Use_RSI_Filter ? "Enabled" : "Disabled");

   return(INIT_SUCCEEDED);
}

//+------------------------------------------------------------------+
//|                      EXPERT DEINITIALIZATION                     |
//+------------------------------------------------------------------+
void OnDeinit(const int reason)
{
   // Release indicator handle
   if(rsi_handle != INVALID_HANDLE)
      IndicatorRelease(rsi_handle);

   Print("Gold Scalping MillX EA deinitialized. Reason: ", reason);
}

//+------------------------------------------------------------------+
//|                        EXPERT TICK FUNCTION                      |
//+------------------------------------------------------------------+
void OnTick()
{
   // Check if trading is allowed
   if(!IsTradeAllowed()) return;

   // Update trading variables
   UpdateTradingInfo();

   // Check for new trade opportunities
   if(ShouldOpenNewTrade())
   {
      ProcessTradingSignals();
   }

   // Monitor existing positions
   MonitorPositions();

   // Check for close all conditions
   CheckCloseAllConditions();
}

//+------------------------------------------------------------------+
//|                       TRADING FUNCTIONS                          |
//+------------------------------------------------------------------+

bool IsTradeAllowed()
{
   // Check trading hours
   if(Use_Trading_Hours)
   {
      MqlDateTime dt;
      TimeToStruct(TimeCurrent(), dt);

      if(dt.hour < Start_Hour || dt.hour > End_Hour)
         return false;
   }

   // Check if market is open
   if(!SymbolInfoInteger(_Symbol, SYMBOL_TRADE_MODE))
      return false;

   // Check maximum drawdown
   if(Enable_Max_DD_Pause && GetCurrentDrawdownPercent() >= Max_DD_Stop_Percent)
   {
      trading_allowed = false;
      return false;
   }

   return trading_allowed;
}

void ProcessTradingSignals()
{
   double rsi_value = GetRSIValue();

   if(rsi_value <= 0) return; // Invalid RSI value

   double current_price = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   double current_bid = SymbolInfoDouble(_Symbol, SYMBOL_BID);

   // Check for BUY signal
   if(ShouldOpenBuy(rsi_value, current_price))
   {
      OpenBuyPosition();
   }

   // Check for SELL signal  
   if(ShouldOpenSell(rsi_value, current_bid))
   {
      OpenSellPosition();
   }
}

bool ShouldOpenBuy(double rsi_value, double price)
{
   // RSI Filter Check
   if(Use_RSI_Filter && rsi_value >= RSI_Overbought)
      return false;

   // Check if this is first buy or if price moved enough for martingale
   if(current_buy_level == 0)
      return true; // First trade

   // Martingale condition: price moved down by minimum distance
   if(last_buy_price > 0 && (last_buy_price - price) >= Min_Distance_Points * _Point)
      return true;

   return false;
}

bool ShouldOpenSell(double rsi_value, double price)
{
   // RSI Filter Check
   if(Use_RSI_Filter && rsi_value <= RSI_Oversold)
      return false;

   // Check if this is first sell or if price moved enough for martingale
   if(current_sell_level == 0)
      return true; // First trade

   // Martingale condition: price moved up by minimum distance
   if(last_sell_price > 0 && (price - last_sell_price) >= Min_Distance_Points * _Point)
      return true;

   return false;
}

void OpenBuyPosition()
{
   if(current_buy_level >= Max_Open_Positions)
      return;

   double lot_size = GetLotSize(ORDER_TYPE_BUY, current_buy_level);
   double price = SymbolInfoDouble(_Symbol, SYMBOL_ASK);
   double sl = (Stop_Loss_Points > 0 && Stop_Loss_Points < 50000) ? price - Stop_Loss_Points * _Point : 0;
   double tp = Take_Profit_Points > 0 ? price + Take_Profit_Points * _Point : 0;

   if(trade.Buy(lot_size, _Symbol, price, sl, tp, EA_Comment))
   {
      last_buy_price = price;
      current_buy_level++;
      last_trade_time = TimeCurrent();

      Print("BUY order opened: Level ", current_buy_level, ", Lot: ", lot_size, ", Price: ", price);
   }
   else
   {
      Print("Failed to open BUY position: ", trade.ResultRetcodeDescription());
   }
}

void OpenSellPosition()
{
   if(current_sell_level >= Max_Open_Positions)
      return;

   double lot_size = GetLotSize(ORDER_TYPE_SELL, current_sell_level);
   double price = SymbolInfoDouble(_Symbol, SYMBOL_BID);
   double sl = (Stop_Loss_Points > 0 && Stop_Loss_Points < 50000) ? price + Stop_Loss_Points * _Point : 0;
   double tp = Take_Profit_Points > 0 ? price - Take_Profit_Points * _Point : 0;

   if(trade.Sell(lot_size, _Symbol, price, sl, tp, EA_Comment))
   {
      last_sell_price = price;
      current_sell_level++;
      last_trade_time = TimeCurrent();

      Print("SELL order opened: Level ", current_sell_level, ", Lot: ", lot_size, ", Price: ", price);
   }
   else
   {
      Print("Failed to open SELL position: ", trade.ResultRetcodeDescription());
   }
}

double GetLotSize(ENUM_ORDER_TYPE order_type, int level)
{
   double lots[];
   int array_size = 0;

   // Select appropriate lot size array based on risk level
   switch(Risk_Level)
   {
      case RISK_LOW:
         ArrayCopy(lots, lot_sizes_low);
         array_size = ArraySize(lot_sizes_low);
         break;
      case RISK_MEDIUM:
         ArrayCopy(lots, lot_sizes_medium);
         array_size = ArraySize(lot_sizes_medium);
         break;
      case RISK_HIGH:
         ArrayCopy(lots, lot_sizes_high);
         array_size = ArraySize(lot_sizes_high);
         break;
      case RISK_VERY_HIGH:
         ArrayCopy(lots, lot_sizes_very_high);
         array_size = ArraySize(lot_sizes_very_high);
         break;
      default:
         ArrayCopy(lots, lot_sizes_low);
         array_size = ArraySize(lot_sizes_low);
         break;
   }

   // Get lot size for current level
   if(level < array_size)
      return lots[level];
   else
      return lots[array_size - 1]; // Use last lot size if level exceeds array
}

double GetRSIValue()
{
   double rsi_buffer[];
   ArraySetAsSeries(rsi_buffer, true);

   if(CopyBuffer(rsi_handle, 0, 0, 2, rsi_buffer) < 2)
   {
      Print("Failed to get RSI values");
      return -1;
   }

   return rsi_buffer[0];
}

void MonitorPositions()
{
   // Monitor for position closures and update levels
   int buy_positions = CountPositions(POSITION_TYPE_BUY);
   int sell_positions = CountPositions(POSITION_TYPE_SELL);

   // Update current levels based on actual positions
   if(buy_positions == 0)
   {
      current_buy_level = 0;
      last_buy_price = 0;
   }

   if(sell_positions == 0)
   {
      current_sell_level = 0;
      last_sell_price = 0;
   }
}

int CountPositions(ENUM_POSITION_TYPE position_type)
{
   int count = 0;

   for(int i = 0; i < PositionsTotal(); i++)
   {
      if(PositionGetTicket(i))
      {
         if(PositionGetString(POSITION_SYMBOL) == _Symbol &&
            PositionGetInteger(POSITION_MAGIC) == Magic_Number &&
            PositionGetInteger(POSITION_TYPE) == position_type)
         {
            count++;
         }
      }
   }

   return count;
}

void CheckCloseAllConditions()
{
   double total_profit = GetTotalProfit();
   double total_lots = GetTotalLots();

   // Calculate dynamic take profit based on total lots
   double dynamic_tp = total_lots * Take_Profit_Points * _Point * SymbolInfoDouble(_Symbol, SYMBOL_TRADE_TICK_VALUE);

   if(total_profit >= dynamic_tp && dynamic_tp > 0)
   {
      CloseAllPositions();
      Print("All positions closed due to profit target reached: ", total_profit);
   }
}

double GetTotalProfit()
{
   double total_profit = 0;

   for(int i = 0; i < PositionsTotal(); i++)
   {
      if(PositionGetTicket(i))
      {
         if(PositionGetString(POSITION_SYMBOL) == _Symbol &&
            PositionGetInteger(POSITION_MAGIC) == Magic_Number)
         {
            total_profit += PositionGetDouble(POSITION_PROFIT);
         }
      }
   }

   return total_profit;
}

double GetTotalLots()
{
   double total_lots = 0;

   for(int i = 0; i < PositionsTotal(); i++)
   {
      if(PositionGetTicket(i))
      {
         if(PositionGetString(POSITION_SYMBOL) == _Symbol &&
            PositionGetInteger(POSITION_MAGIC) == Magic_Number)
         {
            total_lots += PositionGetDouble(POSITION_VOLUME);
         }
      }
   }

   return total_lots;
}

void CloseAllPositions()
{
   for(int i = PositionsTotal() - 1; i >= 0; i--)
   {
      if(PositionGetTicket(i))
      {
         if(PositionGetString(POSITION_SYMBOL) == _Symbol &&
            PositionGetInteger(POSITION_MAGIC) == Magic_Number)
         {
            trade.PositionClose(PositionGetTicket(i));
         }
      }
   }

   ResetTradingVariables();
}

void ResetTradingVariables()
{
   last_buy_price = 0;
   last_sell_price = 0;
   current_buy_level = 0;
   current_sell_level = 0;
   trading_allowed = true;
}

bool ShouldOpenNewTrade()
{
   // Prevent too frequent trading
   if(TimeCurrent() - last_trade_time < 5) // 5 seconds minimum between trades
      return false;

   return true;
}

void UpdateTradingInfo()
{
   // Update last prices based on open positions
   for(int i = 0; i < PositionsTotal(); i++)
   {
      if(PositionGetTicket(i))
      {
         if(PositionGetString(POSITION_SYMBOL) == _Symbol &&
            PositionGetInteger(POSITION_MAGIC) == Magic_Number)
         {
            if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_BUY)
            {
               if(last_buy_price == 0 || PositionGetDouble(POSITION_PRICE_OPEN) < last_buy_price)
                  last_buy_price = PositionGetDouble(POSITION_PRICE_OPEN);
            }
            else if(PositionGetInteger(POSITION_TYPE) == POSITION_TYPE_SELL)
            {
               if(last_sell_price == 0 || PositionGetDouble(POSITION_PRICE_OPEN) > last_sell_price)
                  last_sell_price = PositionGetDouble(POSITION_PRICE_OPEN);
            }
         }
      }
   }
}

double GetCurrentDrawdownPercent()
{
   double account_balance = AccountInfoDouble(ACCOUNT_BALANCE);
   double account_equity = AccountInfoDouble(ACCOUNT_EQUITY);

   if(account_balance <= 0) return 0;

   double drawdown_percent = (account_balance - account_equity) / account_balance * 100;

   return drawdown_percent > 0 ? drawdown_percent : 0;
}

//+------------------------------------------------------------------+
//|                        UTILITY FUNCTIONS                         |
//+------------------------------------------------------------------+

string GetRiskLevelString()
{
   switch(Risk_Level)
   {
      case RISK_LOW: return "Low Risk";
      case RISK_MEDIUM: return "Medium Risk"; 
      case RISK_HIGH: return "High Risk";
      case RISK_VERY_HIGH: return "Very High Risk";
      default: return "Unknown";
   }
}

void PrintTradingStatus()
{
   Print("=== GOLD SCALPING MILLX STATUS ===");
   Print("Buy Positions: ", CountPositions(POSITION_TYPE_BUY));
   Print("Sell Positions: ", CountPositions(POSITION_TYPE_SELL));
   Print("Total Profit: $", GetTotalProfit());
   Print("Current Drawdown: ", GetCurrentDrawdownPercent(), "%");
   Print("RSI Value: ", GetRSIValue());
   Print("Trading Allowed: ", trading_allowed ? "Yes" : "No");
}

//+------------------------------------------------------------------+
